UNIT MATPLUS;
     { Binary and numerical matricies unit.
       May be compiled for DOS or  Windows.
     }
INTERFACE

CONST MAXWATCH= 16;

TYPE

Number= real;
TNumArr= array [1..MAXWATCH] of Number;
PNumArr= ^TNumArr;

TBinArr= array [1..MAXWATCH] of Byte;
PBinArr= ^TBinArr;

PBinVector= ^TBinVector;
TBinVector= object
  N: integer;
  Row: PBinArr;
    constructor Init (Dim: integer);
    destructor  Done;
    procedure   Redim (Dim: integer);
    procedure   Free;
    procedure   Copy (var From: TBinVector);
    function    At   (i: integer): boolean;
    procedure   Put  (i: integer; Ai: boolean);
    function    Norm: integer;
    procedure   AndVector (var v: TBinVector);
    procedure   AndNotVector (var v: TBinVector);
    procedure   OrVector  (var v: TBinVector);
    procedure   XorVector (var v: TBinVector);
    procedure   NotVector;
    procedure   Fill (FillByte: Byte);
    procedure   Zeroize;
    procedure   Random (n1: integer);
    procedure   RandomFrom (n1: integer; var S: TBinVector);
    function    RealSize (Dim: integer): integer;
  end;

TBinVecArr= array [1..MAXWATCH] of TBinVector;
PBinVecArr= ^TBinVecArr;

PBinMatrix= ^TBinMatrix;
TBinMatrix= object
  Col: PBinVecArr;
  N, M: integer;
    constructor Init (NRows, MColumns: integer);
    constructor InitNomem (NRows, MColumns: integer);
    destructor  Done;
    destructor  DoneNomem;
    procedure   Redim (NRows, MColumns: integer);
    function    At   (i, j: integer): boolean;
    procedure   Put  (i, j: integer; Aij: boolean);
    function    RealSize (Dim: integer): integer;
  end;

PMatrix= ^TMatrix;
PVector= ^TVector;
TVector= object
  N: integer;
  Row: PNumArr;
    constructor Init (Dim: integer);
    destructor  Done;
    procedure   Redim (Dim: integer);
    procedure   Free;
    procedure   ReplaceBy (A: PMatrix);
    procedure   Copy (var From: TVector);
    procedure   AddVector (C: double; var v: TVector);
    function    Norm: Number;
    function    Normalize: Number;
    procedure   Zeroize;
    function    RealSize (Dim: integer): integer;
  end;

TVecArr= array [1..MAXWATCH] of TVector;
PVecArr= ^TVecArr;

TMatrix= object
  Col: PVecArr;
  N, M: integer;
  isRef, isTemp: boolean;
    constructor Init (NRows, MColumns: integer);
    constructor Let (A: PMatrix);
    destructor  Done;
    procedure   Redim (NRows, MColumns: integer);
    function    At  (i, j: integer): Number;
    procedure   Put (i, j: integer; Aij: Number);
    procedure   ReplaceBy (A: PMatrix);
    procedure   Copy (A: PMatrix);
    procedure   SubMatrixLines (var X: TMatrix; var Bin: TBinVector);
    procedure   JoinVector (Dim: integer);
    procedure   DeleteVector (j: integer);
    procedure   OrtoNormColumn (j: integer; var V: TVector);
    function    Split (nIter: integer; Delta:real; var X,Y:TMatrix): real;
    function    SplitIfMissed (nIter: integer; Delta:real; var B: TBinMatrix; var X,Y:TMatrix): real;
    procedure   AlignColumns (NRows: integer);
    procedure   Zeroize;
    function    Norm: Number;
    function    LoadFrom (var F: File): boolean;
    procedure   SaveInto (var F: File);
    function    Load (Name: String): boolean;
    procedure   Save (Name: String);
    procedure   Free;
    function    RealSize (Dim: integer): integer;
  end;

  { friends: }
    function    Mult     (A, B: PMatrix): PMatrix;
    function    LinComb  (p: double; A:PMatrix; q:double; B:PMatrix): PMatrix;
    function    OrtoNorm (A: PMatrix; var R: TMatrix): PMatrix;
    function    Proj     (X, A: PMatrix): PMatrix;
    function    Ref      (A: PMatrix): PMatrix;
    function    Inverse  (A: PMatrix): PMatrix;
    function    InvTria  (A: PMatrix): PMatrix;
    function    Tran     (A: PMatrix): PMatrix;
    function    Diag     (var V: TVector): PMatrix;
    function    Vector   (var V: TVector): PMatrix;
    function    Scal (var a, b: TVector): Number;

IMPLEMENTATION
{***************************************************************************}
{           Ŀ
               Binary vector in packed format.                 
            }
constructor TBinVector.Init (Dim: integer);
            begin
            N:= Dim; GetMem (Row, RealSize(N));
            end;

destructor  TBinVector.Done; begin Free; end;

procedure   TBinVector.Free;
            begin
            if Row<>nil then FreeMem (Row, RealSize(N));
            end;

function    TBinVector.RealSize (Dim: integer): integer;
            const szBlock=8;
            begin
            RealSize:= (Dim div (8*szBlock) + 1) * szBlock;
            end;

{           Ŀ
               Redims vector size. Adds 0..0 or truncates data 
               if new dimention don't coinsides with old one.  
            }
procedure   TBinVector.Redim (Dim: integer);
            var i, Nmin: integer;
                V: TBinVector;
            begin
            if RealSize(N)<>RealSize(Dim) then begin
              V.Init (Dim);
              V.Zeroize; V.Copy (Self);
              Free;
              Row:= V.Row; V.Row:= nil;
              V.Done;
              end;
            for i:= N+1 to Dim do Put(i,FALSE);
            N:= Dim;
            end;

procedure   TBinVector.Copy (var From: TBinVector);
            var Nmin: integer;
            begin
            if From.N<N then Nmin:= From.N else Nmin:= N;
            Move (From.Row^, Row^, RealSize(Nmin));
            end;

function    TBinVector.At (i: integer): boolean;
            begin
            At:= FALSE;
            if (i<1) or (i>N) then exit;
            Dec (i);
            At:= Row^[1 + i div 8] and (1 shl byte(7 - i mod 8)) <> 0;
            end;

procedure   TBinVector.Put (i: integer; Ai: boolean);
            var b: ^Byte;
            begin
            if (i<1) or (i>N) then exit;
            Dec (i);
            b:= @Row^[1 + i div 8];
            if Ai
              then b^:= b^ or      (1 shl byte(7 - i mod 8))
              else b^:= b^ and not (1 shl byte(7 - i mod 8));
            end;

function    TBinVector.Norm: integer;
            var i, Nrm: integer;
            begin
            Nrm:=0;
            for i:= 1 to N do if At(i) then Inc(Nrm);
            Norm:= Nrm;
            end;

procedure   TBinVector.AndVector (var v: TBinVector);
            var i: integer;
            begin
            if N=v.N then
              for i:= 1 to N div 8 + 1 do
                Row^[i]:= Row^[i] AND v.Row^[i];
            end;

procedure   TBinVector.AndNotVector (var v: TBinVector);
            var i: integer;
            begin
            if N=v.N then
              for i:= 1 to N div 8 + 1 do
                Row^[i]:= Row^[i] AND NOT v.Row^[i];
            end;

procedure   TBinVector.OrVector (var v: TBinVector);
            var i: integer;
            begin
            if N=v.N then
              for i:= 1 to N div 8 + 1 do
                Row^[i]:= Row^[i] OR v.Row^[i];
            end;

procedure   TBinVector.XorVector (var v: TBinVector);
            var i: integer;
            begin
            if N=v.N then
              for i:= 1 to N div 8 + 1 do
                Row^[i]:= Row^[i] XOR v.Row^[i];
            end;

procedure   TBinVector.NotVector;
            var i: integer;
            begin
            for i:= 1 to N div 8 + 1 do Row^[i]:= NOT Row^[i];
            end;

procedure   TBinVector.Fill (FillByte: Byte);
            begin
            FillChar (Row^, RealSize(N), FillByte);
            end;

procedure   TBinVector.Zeroize;
            begin
            Fill (0);
            end;
{
            Random binary vector with n1 ones
}
procedure   TBinVector.Random (n1: integer);
            var i, iAbs: integer;
            begin
            if n1<1 then n1:=0;
            if n1>N then n1:=N;
            if n1<=N div 2 then begin
              Fill(0);
              for i:= 1 to n1 do begin
                repeat
                  iAbs:= 1 + System.Random (N);
                until not At (iAbs);
                Put (iAbs, TRUE);
                end;
              end
            else begin
              Fill($FF);
              for i:= 1 to N-n1 do begin
                repeat
                  iAbs:= 1 + System.Random (N);
                until At (iAbs);
                Put (iAbs, FALSE);
                end;
              end;
            end;
{
            Random binary vector with n1 ones on places, where S[i]=1.
            Naturally, n1<=S.Norm.
            S and Self are suggested to be of equal sizes, but it is
            not obligatory, procedure will work OK with any sizes.
}
procedure   TBinVector.RandomFrom (n1: integer; var S: TBinVector);
            var i, iAux, Nmin: integer;
                AuxBin: TBinVector;
            begin
            Nmin:= S.Norm;
            if N<Nmin then Nmin:=N;
            if n1<1 then n1:=0;
            if n1>Nmin then n1:= Nmin;
            AuxBin.Init (Nmin);
            AuxBin.Random (n1);
            Fill(0);
            iAux:=1;
            for i:= 1 to S.N do
              if S.At(i) then begin
                if AuxBin.At(iAux) then Put (i,TRUE);
                Inc (iAux);
                end;
            AuxBin.Done;
            end;

{***************************************************************************}
{           Ŀ
               Binary matrix.                                  
            }
constructor TBinMatrix.Init (NRows, MColumns: integer);
            var j: integer;
            begin
            N:= NRows; M:= MColumns;
            GetMem (Col, RealSize(M));
            for j:= 1 to M do Col^[j].Init (N);
            end;

constructor TBinMatrix.InitNomem (NRows, MColumns: integer);
            begin
            N:= NRows; M:= MColumns;
            GetMem (Col, RealSize(M));
            end;

destructor  TBinMatrix.Done;
            var j: integer;
            begin
            for j:= 1 to M do Col^[j].Done;
            FreeMem (Col, RealSize(M));
            end;

destructor  TBinMatrix.DoneNomem;
            begin
            FreeMem (Col, RealSize(M));
            end;

function    TBinMatrix.RealSize (Dim: integer): integer;
            const szBlock=64;
            begin
            RealSize:= (Dim div szBlock + 1) * szBlock * SizeOf(TBinVector);
            end;

procedure   TBinMatrix.Redim (NRows, MColumns: integer);
            var NewCol: PBinVecArr;
                j, min: integer;
            begin
            GetMem (NewCol, RealSize(MColumns));
            if (M<MColumns) then min:=M else min:= MColumns;
            Move (Col^, NewCol^, min*SizeOf(TBinVector));
            for j:= min+1 to MColumns do begin
              NewCol^[j].Init (NRows);
              NewCol^[j].Zeroize;
              end;
            for j:= min+1 to M do Col^[j].Done;
            FreeMem (Col,RealSize(M));
            N:= NRows; M:= MColumns;
            Col:= NewCol;
            for j:= 1 to M do begin
              Col^[j].Redim (N);
              end;
            end;

function    TBinMatrix.At (i, j: integer): boolean;
            begin
            Dec (i);
            At:= Col^[j].Row^[1 + i div 8] and (1 shl byte(7 - i mod 8)) <> 0;
            end;

procedure   TBinMatrix.Put (i, j: integer; Aij: boolean);
            var b: ^Byte;
            begin
            Dec (i);
            b:= @Col^[j].Row^[1 + i div 8];
            if Aij
              then b^:= b^ or      (1 shl byte(7 - i mod 8))
              else b^:= b^ and not (1 shl byte(7 - i mod 8));
            end;

{***************************************************************************}
{           Ŀ
               Kills matrix if it is temporary.                
            }
procedure   MayBeDone (A: PMatrix);
            begin
            if A^.isTemp then Dispose (A, Done);
            end;

{***************************************************************************}
{           Ŀ
               Initializes vector of dimention 'Dim'.          
            }
constructor TVector.Init (Dim: integer);
            begin
            N:= Dim;
            GetMem (Row, RealSize(N));
            end;

{           Ŀ
               Kills vector.                                   
            }
destructor  TVector.Done; begin Free; end;

procedure   TVector.Free;
            begin
            if Row<>nil then FreeMem (Row, RealSize(N));
            end;

{           Ŀ
               Redims vector size. Adds 0..0 or truncates data 
               if new dimention don't coinsides with old one.  
            }
procedure   TVector.Redim (Dim: integer);
            var i, Nmin: integer;
                V: TVector;
            begin
            if RealSize(N)<>RealSize(Dim) then begin
              V.Init (Dim);
              V.Zeroize; V.Copy (Self);
              Free;
              Row:= V.Row; V.Row:= nil;
              V.Done;
              end;
            for i:= N+1 to Dim do Row^[i]:=0;
            N:= Dim;
            end;

{           Ŀ
               Copies vector 'From' into initialized vector.   
               Vectors may have different dimentions.          
            }
procedure   TVector.Copy (var From: TVector);
            var Nmin: integer;
            begin
            if From.N<N then Nmin:= From.N else Nmin:= N;
            Move (From.Row^, Row^, RealSize(Nmin));
            end;

procedure   TVector.ReplaceBy (A: PMatrix);
            begin
            Copy (A^.Col^[1]); MayBeDone (A);
            end;

{           Ŀ
               Adds vector 'v' multiplied by number 'C'.       
               Vectors must be of equal dimention.             
            }
procedure   TVector.AddVector (C: double; var v: TVector);
            var i: integer;
            begin
            if N=v.N then
              for i:= 1 to N do
                Row^[i]:= Row^[i] + C*v.Row^[i];
            end;

{           Ŀ
               Returns scalar product of vectors.              
               Vectors must be of equal dimention.             
            }
function    Scal (var a, b: TVector): Number;
            var Sum: Double;
                i: integer;
            begin
            Sum:=0;
            if a.N=b.N then
              for i:= 1 to a.N do
                Sum:= Sum + a.Row^[i] * b.Row^[i];
            Scal:= Sum;
            end;

{           Ŀ
               Returns Euclidean norm of vector.               
            }
function    TVector.Norm: Number;
            begin
            Norm:= sqrt(Scal(Self,Self));
            end;

{           Ŀ
               Returns Euclidian norm of vector and divide     
               vector by calculated number.                    
            }
function    TVector.Normalize: Number;
            var Sum: Double;
                i: integer;
            begin
            Sum:= Norm;
            if Sum<>0 then
              for i:= 1 to N do Row^[i]:= Row^[i] / Sum;
            Normalize:= Sum;
            end;

{           Ŀ
               Zeroizes vector.                                
            }
procedure   TVector.Zeroize;
            begin
            FillChar (Row^, RealSize(N), 0);
            end;

function    TVector.RealSize (Dim: integer): integer;
            const szBlock=10;
            begin
            RealSize:= ((Dim-1) div szBlock + 1) * szBlock * SizeOf(Number);
            end;

{***************************************************************************}
{           Ŀ
               Initializes matrix of dimention                 
               'NRows' by 'MColumns'.                          
            }
constructor TMatrix.Init (NRows, MColumns: integer);
            var i: integer;
            begin
            N:= NRows;
            M:=0;
            isTemp:= FALSE;
            isRef:= FALSE;
            GetMem (Col, RealSize(M));
            Redim (NRows, MColumns);
            end;

function    TMatrix.RealSize (Dim: integer): integer;
            const szBlock=64;
            begin
            RealSize:= (Dim div szBlock + 1) * szBlock * SizeOf(TVector);
            end;

procedure   TMatrix.Redim (NRows, MColumns: integer);
            var NewCol: PVecArr;
                j, min: integer;
            begin
            GetMem (NewCol, RealSize(MColumns));
            if (M<MColumns) then min:=M else min:= MColumns;
            Move (Col^, NewCol^, min*SizeOf(TVector));
            for j:= min+1 to MColumns do begin
              NewCol^[j].Init (NRows);
              NewCol^[j].Zeroize;
              end;
            for j:= min+1 to M do Col^[j].Done;
            FreeMem (Col, RealSize(M));
            N:= NRows; M:= MColumns;
            Col:= NewCol;
            for j:= 1 to M do Col^[j].Redim (N);
            end;

{           Ŀ
               Initializes matrix with temporary matrix 'A'.   
            }
constructor TMatrix.Let (A: PMatrix);
            begin
            Init (0,0); ReplaceBy (A);
            end;

{           Ŀ
               Assigns new value 'A' to the initialized matrix.
               Kills temporary matrix 'A' automatically.       
            }
procedure   TMatrix.ReplaceBy (A: PMatrix);
            begin
            Free;
            Self:= A^;
            isTemp:= FALSE;
            if A^.isTemp then Dispose (A) else A^.isRef:= TRUE;
            end;

{           Ŀ
               Like to 'ReplaceBy' defines new matrix.         
               But old matrix 'A' remains valid and must be    
               destructed later.                               
            }
procedure   TMatrix.Copy (A: PMatrix);
            var i, j: integer;
            begin
            Redim (A^.N,A^.M);
            for i:= 1 to A^.N do
              for j:= 1 to A^.M do
                Col^[j].Row^[i]:= A^.Col^[j].Row^[i];
            isTemp:= FALSE;
            isRef:= FALSE;
            end;

{           Ŀ
               Creates temporary matrix with fixed dimention.  
            }
function    Temp (NRows, MColumns: integer): PMatrix;
            var A: PMatrix;
            begin
            New (A, Init (NRows, MColumns));
            A^.isTemp:= TRUE;
            Temp:= A;
            end;

{           Ŀ
               At and Put facilitate access to matrix elements.
            }
function    TMatrix.At (i, j: integer): Number;
            begin
            At:= Col^[j].Row^[i];
            end;

procedure   TMatrix.Put (i, j: integer; Aij: Number);
            begin
            Col^[j].Row^[i]:= Aij;
            end;

{           Ŀ
               Returns Euclidean norm of matrix.               
            }
function    TMatrix.Norm: Number;
            var Sum: double;
                j: integer;
            begin
            Sum:= 0;
            for j:= 1 to M do Sum:= Sum + Scal (Col^[j], Col^[j]);
            Norm:= sqrt (Sum);
            end;

{           Ŀ
               Returns (temporary) product of 'A' by 'B'.      
               Kills temporary matricies automatically.        
            }
function    Mult (A, B: PMatrix): PMatrix;
            var i, j, k: integer;
                Sum: Double;
                C: PMatrix;
            begin
            if A^.M=B^.N then begin
              C:= Temp (A^.N, B^.M);
              for i:= 1 to A^.N do for j:= 1 to B^.M do begin
                Sum:=0;
                for k:= 1 to A^.M do
                  Sum:= Sum + A^.Col^[k].Row^[i] * B^.Col^[j].Row^[k];
                C^.Col^[j].Row^[i]:= Sum;
                end;
              end;
            MayBeDone (A);
            MayBeDone (B);
            Mult:= C;
            end;

{           Ŀ
               Returns (temporary) linear combination of A     
               and B. Kills temporary matricies automatically. 
            }
function    LinComb (p:double; A:PMatrix; q:double; B:PMatrix): PMatrix;
            var i,j: integer;
                C: PMatrix;
            begin
            if (A^.N=B^.N) and (A^.M=B^.M) then begin
              C:= Temp (A^.N, A^.M);
              for i:= 1 to A^.N do for j:= 1 to A^.M do
                C^.Col^[j].Row^[i]:=
                  p*A^.Col^[j].Row^[i] + q*B^.Col^[j].Row^[i];
              end;
            MayBeDone (A);
            MayBeDone (B);
            LinComb:= C;
            end;

{           Ŀ
               Forms submatrix of 'X', containing selected     
               lines accordingly binary vector 'Bin'.          
            }
procedure   TMatrix.SubMatrixLines (var X: TMatrix; var Bin: TBinVector);
            var i, j, iSub, Nmin, Nsub: integer;
            begin
            Nmin:= Bin.N; if X.N<Nmin then Nmin:=X.N;
            Nsub:= Bin.Norm;
            if Nsub>Nmin then Nsub:=Nmin;
            Redim (Nsub,X.M);
            iSub:=0;
            for i:=1 to Nmin do
              if Bin.At(i) then begin
                Inc (iSub);
                for j:=1 to X.M do
                  Col^[j].Row^[iSub]:= X.Col^[j].Row^[i];
                end;
            end;

{           Ŀ
               Returns (temporary) projection of matrix 'A' by 
               orthogonal complementation of ORTHOGONAL 'X'.   
            }
function    Proj (X, A: PMatrix): PMatrix;
            var C: PMatrix;
            begin
            if (X^.M=0) or (X^.N<>A^.N) then
              Proj:= LinComb (1,A, 0, Ref(A)) else
              Proj:= LinComb (1,A, -1,Mult (X, Mult(Tran(Ref(X)), Ref(A))));
            end;

{           Ŀ
               Returns the copy of (reference to) matrix 'A'.  
               Allows to bypass temporary matricies disposing. 
            }
function    Ref (A: PMatrix): PMatrix;
            var B: PMatrix;
            begin
            New(B);
            Move (A^, B^, SizeOf(TMatrix));
            B^.isRef:= TRUE;
            B^.isTemp:= TRUE;
            Ref:= B;
            end;

{           Ŀ
               Returns the inverse matrix for square matrix 'A'
               and left pseudo-inverse for rectangular one.    
            }
function    Inverse (A: PMatrix): PMatrix;
            var C: PMatrix; R: TMatrix;
            begin
            C:= OrtoNorm (A,R);
            Inverse:= Mult (InvTria(@R), Tran(C));
            R.Done;
            end;

{           Ŀ
                Returns the inverse matrix for                 
                upper triangular matrix 'A'.                   
            }
function    InvTria (A: PMatrix): PMatrix;
            var i,j,k: integer;
                Sum: Double;
                C: PMatrix;
            begin
            if A^.N=A^.M then begin
              C:= Temp (A^.N, A^.N);
              with C^ do begin
                for i:= 1 to N do Col^[i].Row^[i]:= 1.0/A^.Col^[i].Row^[i];
                for i:= N downto 1 do
                  for j:= i+1 to N do begin
                    Sum:=0;
                    for k:= i+1 to j do
                      Sum:= Sum + A^.Col^[k].Row^[i] * Col^[j].Row^[k];
                    Col^[j].Row^[i]:= -Sum / A^.Col^[i].Row^[i];
                    Col^[i].Row^[j]:= 0;
                    end;
                end;
              end;
            MayBeDone (A);
            InvTria:= C;
            end;

{           Ŀ
               Calculates k-splitting (X,Y) of matrix.         
               Don't initialize matricies 'X' and 'Y'!         
            }
function    TMatrix.Split (nIter: integer; Delta: real; var X,Y: TMatrix): real;
            var k, r, j: integer;
                NormZ: Double;
                Norms, Waste: TVector;
            begin
            X.Init (N,0); Y.Init (M,0);
            Waste.Init (M);
            Norms.Init (M); NormZ:= 0;
            for j:= 1 to M do begin
              Norms.Row^[j]:= Scal (Col^[j], Col^[j]);
              NormZ:= NormZ + Norms.Row^[j];
              end;
            while (NormZ/N/M > sqr(Delta)) and (X.M<N) and (X.M<M) do begin
              X.JoinVector (N); k:= X.M;
              Y.JoinVector (M);
              Y.Col^[k].Copy (Norms);
              for r:= 1 to nIter do begin
                X.Col^[k].ReplaceBy (Mult (@Self, Vector (Y.Col^[k])));
                X.OrtoNormColumn (k, Waste);
                Y.Col^[k].ReplaceBy (Tran (Mult (Tran (Vector (X.Col^[k])), @Self)));
                end;
              for j:= 1 to M do begin
                Norms.Row^[j]:= Norms.Row^[j] - sqr (Y.Col^[k].Row^[j]);
                NormZ:= NormZ - sqr (Y.Col^[k].Row^[j]);
                end;
              end;
            Waste.Done; Norms.Done;
            Split:= sqrt (abs(NormZ/N/M));
            end;

{           Ŀ
               Calculates k-splitting (X,Y) of matrix with     
               missed elements. Binary matrix 'B' indicates:   
               Bij=0 if Zij is missed, Bij=1 if Zij present.   
               Don't initialize matricies 'X' and 'Y'!         
            }
function    TMatrix.SplitIfMissed (nIter: integer; Delta: real; var B: TBinMatrix; var X,Y: TMatrix): real;
            var k, r, i, j, s: integer;
                Count: longint;
                NormZ, PrecZ, Sum: Double;
                Norms: TVector;

          procedure CalcNorms;
            var i, j, s: integer;
            begin
            Norms.Zeroize; NormZ:= 0.0; Count:= 0;
            for j:= 1 to M do
              for i:= 1 to N do
                if B.At(i,j) then begin
                  Sum:= 0.0;
                  for s:= k downto 1 do
                    Sum:= Sum + X.Col^[s].Row^[i] * Y.Col^[s].Row^[j];
                  Sum:= sqr (Col^[j].Row^[i]-Sum);
                  Norms.Row^[j]:= Norms.Row^[j] + Sum;
                  NormZ:= NormZ + Sum;
                  Inc (Count);
                  end;
            end;

          procedure FillMissed;
            var i, j, s: integer;
            begin
            for j:= 1 to M do
              for i:= 1 to N do
                if not B.At(i,j) then begin
                  Sum:= 0.0;
                  for s:= k downto 1 do
                    Sum:= Sum + X.Col^[s].Row^[i] * Y.Col^[s].Row^[j];
                  Col^[j].Row^[i]:= Sum;
                  end;
            end;

          procedure Iteration (k: integer; DoFill: boolean);
            var r: integer;
                Waste: TVector;
            begin
            Waste.Init (M);
            for r:= 1 to nIter do begin
              X.Col^[k].ReplaceBy (Mult (@Self, Vector (Y.Col^[k])));
              X.OrtoNormColumn (k, Waste);
              Y.Col^[k].ReplaceBy (Tran (Mult (Tran (Vector (X.Col^[k])), @Self)));
              if DoFill then FillMissed;
              end;
            Waste.Done;
            end;

            begin
            X.Init (N,0); Y.Init (M,0); k:=0;
            Norms.Init (M);
            CalcNorms;
            {ShowBinMatrix (B, str(N*1.0*M-Count)+' points from '+str(N*1.0*M)+' are missed:');}
            repeat
              X.JoinVector (N);
              Y.JoinVector (M);
              k:= X.M;
              Y.Col^[k].Copy (Norms);
              for r:= 1 to k do
                for s:= 1 to k do Iteration (s, s=k);
              CalcNorms;
            until (NormZ/Count < sqr(Delta)) or (k>N) or (k>M);
            Norms.Done;
            SplitIfMissed:= sqrt (abs(NormZ/Count));
            end;

{           Ŀ
               Returns the transposed matrix.                  
            }
function    Tran (A: PMatrix): PMatrix;
            var C: PMatrix;
                i, j: integer;
            begin
            C:= Temp (A^.M, A^.N);
            for i:= 1 to A^.N do for j:= 1 to A^.M do
              C^.Col^[i].Row^[j]:= A^.Col^[j].Row^[i];
            MayBeDone (A);
            Tran:= C;
            end;

{           Ŀ
               Joins column vector if dimention 'Dim' to the   
               right edge of matrix.                           
            }
procedure   TMatrix.JoinVector (Dim: integer);
            var max: integer;
            begin
            if N<Dim then max:=Dim else max:=N;
            ReDim (max, M+1);
            end;

{           Ŀ
               Deletes column number 'j' from the matrix.      
               Shifts successives columns to the left.         
            }
procedure   TMatrix.DeleteVector (j: integer);
            var NewCol: PVecArr;
                k: integer;
            begin
            Col^[j].Done;
            for k:= j to M-1 do Col^[k]:= Col^[k+1];
            if RealSize(M)<>RealSize(M-1) then begin
              GetMem (NewCol, RealSize(M-1));
              Move (Col^, NewCol^, (M-1)*SizeOf(TVector));
              FreeMem (Col, RealSize(M));
              Col:= NewCol;
              end;
            Dec (M);
            end;

{           Ŀ
               Alignes columns of matrix to unique dimention   
               'NRows'. Adds zeros if column was shorter.      
            }
procedure   TMatrix.AlignColumns (NRows: integer);
            begin
            Redim (NRows, M);
            end;

{           Ŀ
               Orthonormalize column 'j' with respect to       
               precedent columns (just orthonormalized).       
               Records transformation coefficients into 'V'.   
               Vector 'V' must be Init with dimention 'j'.     
            }
procedure   TMatrix.OrtoNormColumn (j: integer; var V: TVector);
            var i, k: integer;
            begin
            (*
          { Classical orthogonalization: }
            for k:= 1 to j-1 do
              V.Row^[k]:= Scal (Col^[k], Col^[j]);
            for k:= 1 to j-1 do
              Col^[j].AddVector (-V.Row^[k], Col^[k]); *)

          { Modified orthogonalization: }
            for k:= 1 to j-1 do begin
              V.Row^[k]:= Scal (Col^[k], Col^[j]);
              Col^[j].AddVector (-V.Row^[k], Col^[k]);
              end;
            V.Row^[j]:= Col^[j].Normalize;
            end;

{           Ŀ
               Returns orthonormalized matrix 'A' and records  
               transformation matrix into 'R'.                 
               Don't initialize matrix 'R'!                    
            }
function    OrtoNorm (A: PMatrix; var R: TMatrix): PMatrix;
            var j: integer;
                C: PMatrix;
            begin
            R.Init (0,0);
            C:= Temp (A^.N, A^.M);
            for j:= 1 to A^.M do begin
              R.JoinVector (j);
              C^.Col^[j].Copy (A^.Col^[j]);
              C^.OrtoNormColumn (j, R.Col^[j]);
              end;
            R.AlignColumns (A^.M);
            MayBeDone (A);
            OrtoNorm:= C;
            end;

{           Ŀ
               Returns diagonal matrix with diagonal 'V'.      
            }
function    Diag (var V: TVector): PMatrix;
            var A: PMatrix;
                j: integer;
            begin
            A:= Temp (V.N, V.N);
            for j:= 1 to A^.M do A^.Col^[j].Zeroize;
            for j:= 1 to A^.M do A^.Col^[j].Row^[j]:= V.Row^[j];
            Diag:= A;
            end;


{           Ŀ
               Returns matrix with one column 'V'.             
            }
function    Vector (var V: TVector): PMatrix;
            var A: PMatrix;
            begin
            A:= Temp (V.N, 1);
            A^.Col^[1].Copy (V);
            Vector:= A;
            end;

{           Ŀ
               Zeroizes matrix.                                
            }
procedure   TMatrix.Zeroize;
            var j: integer;
            begin
            for j:= 1 to M do Col^[j].Zeroize;
            end;

{           Ŀ
               Loads matrix from data file 'F'.                
            }
function    TMatrix.LoadFrom (var F: File): boolean;
            var NumRead, NN, MM, j: integer;
            begin
            LoadFrom:= FALSE;
            BlockRead (F, NN, SizeOf(NN), NumRead);
            if NumRead=SizeOf(NN) then begin
              BlockRead (F, MM, SizeOf(MM), NumRead);
              if NumRead=SizeOf(MM) then begin
                Free;
                Init (NN, MM);
                for j:= 1 to M do begin
                  BlockRead (F, Col^[j].Row^, N*SizeOf(Number), NumRead);
                  if (NumRead<>N*SizeOf(Number)) then exit;
                  end;
                LoadFrom:= TRUE;
                end;
              end;
            end;

function    TMatrix.Load (Name: String): boolean;
            var F: File;
            begin
            assign (F, Name);
            {$I-} Reset (F,1); {$I-}
            if IOResult=0 then begin
              Load:= LoadFrom (F); close (F);
              end
            else Load:= FALSE;
            end;

{           Ŀ
               Saves matrix into data file 'F'.                
            }
procedure   TMatrix.SaveInto (var F: File);
            var j, Res: integer;
            begin
            BlockWrite (F, N, SizeOf(N), Res);
            BlockWrite (F, M, SizeOf(M), Res);
            for j:= 1 to M do
              BlockWrite (F, Col^[j].Row^, N*SizeOf(Number), Res);
            end;

procedure   TMatrix.Save (Name: String);
            var F: File;
            begin
            assign (F, Name);
            ReWrite (F,1);
            SaveInto (F);
            close (F);
            end;

{           Ŀ
               Frees memory allocated for matrix.              
            }
procedure   TMatrix.Free;
            var j: integer;
            begin
            if (Col<>nil) and not isRef then begin
              for j:= 1 to M do Col^[j].Done;
              FreeMem (Col, RealSize(M));
              end;
            Col:= nil;
            end;

{           Ŀ
               Kills matrix.                                   
            }
destructor  TMatrix.Done;
            begin
            Free;
            end;
END.